<?php
ini_set('display_errors', 1);
error_reporting(E_ALL);
session_start();
include 'db.php';

if (!isset($_SESSION['phone'])) {
    header("Location: login.php");
    exit();
}

$phone = $_SESSION['phone'];
$success = $error = "";

// Fetch user info
$stmt = $conn->prepare("SELECT id, flexible_wallet FROM users WHERE phone = ?");
$stmt->execute([$phone]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);
$user_id = $user['id'];
$flexible_wallet = $user['flexible_wallet'];

// Fetch wallets
$wallet_stmt = $conn->prepare("SELECT * FROM wallets WHERE phone = ?");
$wallet_stmt->execute([$phone]);
$wallets = $wallet_stmt->fetchAll(PDO::FETCH_ASSOC);

if (count($wallets) === 0) {
    header("Location: wallet_address.php");
    exit();
}

// Make array for JS mapping
$wallet_map = [];
foreach ($wallets as $w) {
    $wallet_map[$w['network']] = $w['address'];
}

$default_network = $wallets[0]['network'];
$default_address = $wallets[0]['address'];

// Handle withdrawal submission
$today = date("Y-m-d");
$stmt = $conn->prepare("SELECT COUNT(*) FROM withdrawals WHERE user_id=? AND DATE(created_at)=?");
$stmt->execute([$user_id, $today]);
$today_withdrawals = $stmt->fetchColumn();

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $amount = floatval($_POST['amount']);
    $address = trim($_POST['address']);
    $currency = $_POST['currency'] ?? 'USDT';
    $mainnet = $_POST['mainnet'] ?? 'TRC20';

    if ($today_withdrawals > 0) {
        $error = "Only one withdrawal is allowed per day.";
    } elseif ($amount < 10 || $amount % 10 != 0) {
        $error = "Minimum withdrawal is 10 USDT and must be multiples of 10.";
    } elseif ($amount > $flexible_wallet) {
        $error = "Insufficient balance.";
    } elseif (empty($address)) {
        $error = "Please enter a withdrawal address.";
    } else {
        // Fee calculation
        $commission = 0;
        if ($amount >= 10 && $amount <= 100) $commission = $amount * 0.08;
        elseif ($amount >= 110 && $amount <= 500) $commission = $amount * 0.05;
        elseif ($amount >= 510 && $amount <= 1000) $commission = $amount * 0.01;

        $income_tax = 1;
        $total_fee = $commission + $income_tax;
        $net_amount = $amount - $total_fee;

        if ($net_amount <= 0) {
            $error = "Withdrawal amount after fees must be greater than 0.";
        } else {
            $stmt = $conn->prepare("INSERT INTO withdrawals 
                (user_id, phone, address, amount, fee, net_amount, type, mainnet, status, requested_at) 
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())");
            $stmt->execute([$user_id, $phone, $address, $amount, $total_fee, $net_amount, $currency, $mainnet]);

            $stmt = $conn->prepare("UPDATE users SET flexible_wallet = flexible_wallet - ? WHERE id=?");
            $stmt->execute([$amount, $user_id]);
            $success = "Withdrawal submitted successfully.";
        }
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<title>Asset Withdrawal</title>
<style>
body { background:#000; color:#fff; font-family:'Poppins',sans-serif; margin:0; }
.topbar { display:flex; align-items:center; padding:15px; font-size:18px; }
.topbar a { color:#fff; text-decoration:none; font-size:22px; }
.topbar span { flex:1; text-align:center; }

.balance-card {
  background: linear-gradient(180deg, #FFD57E 0%, #C29A51 100%);
  color:#000; text-align:center; border-radius:12px;
  margin:20px 15px; padding:25px 0; font-weight:bold;
}
.balance-card small { display:block; font-size:13px; color:#4b3a13; }
.balance-card h1 { font-size:26px; margin:5px 0 0; }

.section { background:#111; border-radius:12px; padding:20px; margin:15px; }
.label { font-size:14px; color:#bbb; margin-bottom:8px; }
.selector {
  width:100%; background:#222; color:#fff; border:none; border-radius:8px;
  padding:14px; text-align:left; font-size:15px; cursor:pointer; position:relative;
}
.selector::after { content:'›'; position:absolute; right:15px; top:50%;
  transform:translateY(-50%) rotate(90deg); color:#888; font-size:18px; }
input[type="number"], input[type="text"] {
  width:100%; padding:14px; background:#222; color:white; border:none;
  border-radius:8px; margin-top:5px; font-size:15px; box-sizing:border-box;
}
.info { font-size:13px; color:#bbb; margin-top:10px; }
.footer-card {
  background:linear-gradient(180deg,#FFD57E 0%,#C29A51 100%);
  color:#000; border:none; border-radius:30px; width:90%;
  margin:20px 5%; padding:15px; font-weight:bold; font-size:16px; cursor:pointer;
}

/* Modal */
.modal {
  position:fixed; left:0; right:0; bottom:-100%; background:#000; color:#fff;
  border-radius:16px 16px 0 0; transition:bottom 0.4s ease; padding:20px; z-index:1000;
}
.modal.show { bottom:0; }
.modal h3 { text-align:center; margin-bottom:10px; }
.modal button.option {
  width:100%; background:#111; color:#fff; border:none; padding:12px; margin:5px 0;
  border-radius:8px; cursor:pointer;
}

/* Toast */
.toast {
  position:fixed; top:50%; left:50%; transform:translate(-50%,-50%);
  background:rgba(0,0,0,0.9); padding:12px 18px; color:#fff;
  border-radius:8px; font-size:13px; z-index:9999; opacity:0; transition:opacity 0.5s;
}
.toast.show { opacity:1; }
</style>
</head>
<body>

<div class="topbar">
  <a href="mine.php">←</a>
  <span>Asset Withdrawal</span>
</div>

<div class="balance-card">
  <small>Currently available assets (USDT)</small>
  <h1><?= number_format($flexible_wallet, 2) ?></h1>
</div>

<div class="section">
<form method="post">
  <div class="label">Select Currency</div>
  <button type="button" id="currencyBtn" class="selector" onclick="openModal('currency')">USDT</button>
  <input type="hidden" name="currency" id="currencyValue" value="USDT">

  <div class="label">Mainnet</div>
  <button type="button" id="mainnetBtn" class="selector" onclick="openModal('mainnet')"><?= htmlspecialchars($default_network) ?></button>
  <input type="hidden" name="mainnet" id="mainnetValue" value="<?= htmlspecialchars($default_network) ?>">

  <div class="label">Withdrawal address</div>
  <input type="text" name="address" id="addressInput" value="<?= htmlspecialchars($default_address) ?>" readonly>

  <div class="label">Withdrawal amount</div>
  <input type="number" name="amount" id="amount" placeholder="Please enter the withdrawal amount">

  <div class="info">ⓘ Withdrawal Amount: 10 USDT - 50000 USDT</div>
  <div class="info" id="feeInfo">Fees: 0 USDT</div>
  <div class="info" id="taxInfo">Income Tax: 0 USDT</div>
  <div class="info" id="actualInfo">Actual arrival: 0 USDT</div>

  <button type="submit" class="footer-card">Confirm withdrawal</button>
</form>
</div>

<!-- Modal -->
<div id="modal" class="modal">
  <h3>Select Option</h3>
  <div id="modalContent"></div>
</div>

<?php if ($error): ?>
<div id="toast" class="toast"><?= htmlspecialchars($error) ?></div>
<?php elseif ($success): ?>
<div id="toast" class="toast"><?= htmlspecialchars($success) ?></div>
<?php endif; ?>

<script>
const walletMap = <?= json_encode($wallet_map) ?>;

function openModal(type){
  const modal=document.getElementById('modal');
  const content=document.getElementById('modalContent');
  content.innerHTML='';
  let options=[];
  if(type==='currency') options=['USDT','BTC','ETH'];
  else if(type==='mainnet') options=Object.keys(walletMap);

  options.forEach(opt=>{
    const btn=document.createElement('button');
    btn.textContent=opt;
    btn.className='option';
    btn.onclick=()=>{
      if(type==='currency'){
        document.getElementById('currencyBtn').textContent=opt;
        document.getElementById('currencyValue').value=opt;
      }else if(type==='mainnet'){
        document.getElementById('mainnetBtn').textContent=opt;
        document.getElementById('mainnetValue').value=opt;
        document.getElementById('addressInput').value=walletMap[opt]||'';
      }
      modal.classList.remove('show');
    };
    content.appendChild(btn);
  });
  modal.classList.add('show');
}

document.getElementById('amount').addEventListener('input',()=>{
  let amount=parseFloat(document.getElementById('amount').value)||0;
  let fee=0,incomeTax=1;
  if(amount>=10&&amount<=100)fee=amount*0.08;
  else if(amount>=110&&amount<=500)fee=amount*0.05;
  else if(amount>=510&&amount<=1000)fee=amount*0.01;
  let total=fee+incomeTax;
  let net=amount-total;
  document.getElementById('feeInfo').innerText="Fees: "+fee.toFixed(2)+" USDT";
  document.getElementById('taxInfo').innerText="Income Tax: "+incomeTax.toFixed(2)+" USDT";
  document.getElementById('actualInfo').innerText="Actual arrival: "+(net>0?net.toFixed(2):"0.00")+" USDT";
});

window.addEventListener("DOMContentLoaded",()=>{
  const toast=document.getElementById("toast");
  if(toast){
    setTimeout(()=>toast.classList.add("show"),100);
    setTimeout(()=>toast.classList.remove("show"),3000);
  }
});
</script>
</body>
</html>