 <?php
// watch_task.php — Full updated version
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
session_start();
include('db.php');

if (!isset($_SESSION['phone'])) {
    header("Location: login.php");
    exit();
}

$phone   = $_SESSION['phone'];
$task_id = isset($_GET['task_id']) ? (int)$_GET['task_id'] : 0;

if ($task_id < 1) {
    echo "Invalid video selected.";
    exit();
}


/*
 * Ensure assets table exists and supports community_income type
 */
$conn->exec(<<<'SQL'
CREATE TABLE IF NOT EXISTS `assets` (
    `id` INT UNSIGNED NOT NULL AUTO_INCREMENT PRIMARY KEY,
    `user_id` INT UNSIGNED NOT NULL,
    `type` ENUM('task_income','community_income') NOT NULL,
    `amount` DECIMAL(18,8) NOT NULL DEFAULT 0,
    `description` VARCHAR(255) NOT NULL,
    `created_at` DATETIME NOT NULL DEFAULT CURRENT_TIMESTAMP,
    INDEX(`user_id`)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4;
SQL
);

// load user
$stmt = $conn->prepare("SELECT * FROM users WHERE phone = ?");
$stmt->execute([$phone]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$user) {
    echo "User not found.";
    exit();
}
$user_id = (int)$user['id'];

// fetch video
$stmt = $conn->prepare("SELECT * FROM videos WHERE id = ?");
$stmt->execute([$task_id]);
$video = $stmt->fetch(PDO::FETCH_ASSOC);
if (!$video) {
    echo "Video not found.";
    exit();
}
$video_vip_required = (int)$video['vip_product_id']; // vip requirement assigned to the video

/*
 * If video_vip_required === 0 => VIP0 (free)
 * For other videos we check user's purchase for that VIP (later) & daily limits.
 */

/* -------------------------
 * Determine user's VIP purchase record for that vip_product (if any)
 * ------------------------- */
$vip_id = $video_vip_required;
$vip = null;
if ($vip_id === 0) {
    // VIP 0 config (if you keep a vip_products row for 0)
    $stmt = $conn->prepare("SELECT * FROM vip_products WHERE id = 0 LIMIT 1");
    $stmt->execute();
    $vip = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$vip) {
        // fallback defaults for VIP0
        $vip = [
            'daily_earning_limit' => 999999,
            'valid_days' => 999999,
            'viewing_income' => 0.0,
            'level' => 0
        ];
    }
} else {
    // Get latest purchase row for this user & vip product
    $stmt = $conn->prepare("
        SELECT p.*, v.name, v.daily_earning_limit, v.valid_days, v.level, v.viewing_income
        FROM purchases p
        JOIN vip_products v ON p.vip_product_id = v.id
        WHERE p.user_id = ? AND p.vip_product_id = ?
        ORDER BY p.purchased_at DESC
        LIMIT 1
    ");
    $stmt->execute([$user_id, $vip_id]);
    $vip = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$vip) {
        echo "You have not purchased this VIP package.";
        exit();
    }
    // check expiry if needed (some codebases store expire_at in purchases)
    if (!empty($vip['expire_at'])) {
        $now = new DateTime();
        $expire = new DateTime($vip['expire_at']);
        if ($now > $expire) {
            echo "Your VIP access has expired.";
            exit();
        }
    }
}

/* -------------------------
 * Count tasks scored today for this user & this vip product
 * ------------------------- */
$stmt = $conn->prepare("SELECT COUNT(*) FROM task_records WHERE user_id = ? AND vip_product_id = ? AND DATE(scored_at) = CURDATE()");
$stmt->execute([$user_id, $vip_id]);
$scored_today = (int)$stmt->fetchColumn();

/* -------------------------
 * Fetch all videos for UI (not used in backend logic below)
 * ------------------------- */
$stmt = $conn->prepare("SELECT * FROM videos WHERE vip_product_id = ?");
$stmt->execute([$vip_id]);
$all_videos = $stmt->fetchAll();

/* -------------------------
 * Already scored this video today?
 * ------------------------- */
$stmt = $conn->prepare("SELECT 1 FROM task_records WHERE user_id = ? AND video_id = ? AND DATE(scored_at) = CURDATE() LIMIT 1");
$stmt->execute([$user_id, $task_id]);
$already_scored = (bool)$stmt->fetchColumn();

/* -------------------------
 * Handle scoring POST
 * ------------------------- */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && !$already_scored && $scored_today < (int)$vip['daily_earning_limit']) {
    try {
        // Start transaction
        $conn->beginTransaction();

        // Determine income based on business rules (kept same as your prior logic)
        if ($vip_id === 0) {
            $income = (float)$vip['viewing_income'];
        } elseif ($vip_id >= 3) {
            $income = 1.0;
        } else {
            $income = 0.6;
        }

        // 1) Credit performer flexible_wallet
        $stmt = $conn->prepare("UPDATE users SET flexible_wallet = flexible_wallet + ? WHERE id = ?");
        $stmt->execute([$income, $user_id]);

        // 2) Insert task record
        $stmt = $conn->prepare("INSERT INTO task_records (user_id, video_id, vip_product_id, scored_at, amount) VALUES (?, ?, ?, NOW(), ?)");
        $stmt->execute([$user_id, $task_id, $vip_id, $income]);

        // 3) Log performer task income in assets
        $stmt = $conn->prepare("INSERT INTO assets (user_id, type, amount, description, created_at) VALUES (?, 'task_income', ?, ?, NOW())");
        $stmt->execute([$user_id, $income, "Video Task Income (Video ID: $task_id)"]);

        /*
         * COMMUNITY REBATES RULES:
         * - The task must be a VIP >= 1 task (i.e. the video is tied to VIP >= 1) OR
         *   you prefer to check the performing user's highest VIP level.
         * - Per your request: "rebate should only credit VIP 1 and above task".
         *
         * We'll check the performing user's highest purchased vip_products.level and only
         * distribute rebates if that level >= 1.
         */

        // Check performing user's highest VIP level (across all purchases)
        $stmt = $conn->prepare("
            SELECT COALESCE(MAX(v.level), 0) AS max_level
            FROM purchases p
            JOIN vip_products v ON p.vip_product_id = v.id
            WHERE p.user_id = ?
        ");
        $stmt->execute([$user_id]);
        $performerMaxLevel = (int)$stmt->fetchColumn();

        // Only distribute community rebates if the performer has VIP level >= 1
        if ($performerMaxLevel >= 1) {
            // rebates percentages for levels 1..3
            $rebates = [0.06, 0.03, 0.01];

            // walk uplines using invite codes
            $current_invite_code = $user['invited_by'];

            foreach ($rebates as $levelIndex => $percent) {
                if (empty($current_invite_code)) break;

                // get the upline user (by invite_code)
                $stmt = $conn->prepare("SELECT id, invited_by FROM users WHERE invite_code = ? LIMIT 1");
                $stmt->execute([$current_invite_code]);
                $ref_user = $stmt->fetch(PDO::FETCH_ASSOC);
                if (!$ref_user) break;

                $uplineId = (int)$ref_user['id'];

                // check upline's highest VIP level (must be >=1 to receive community rebate)
                $stmt2 = $conn->prepare("
                    SELECT COALESCE(MAX(v.level), 0) AS max_level
                    FROM purchases p
                    JOIN vip_products v ON p.vip_product_id = v.id
                    WHERE p.user_id = ?
                ");
                $stmt2->execute([$uplineId]);
                $uplineMaxLevel = (int)$stmt2->fetchColumn();

                if ($uplineMaxLevel >= 1) {
                    // calculate rebate amount
                    $rebate_amount = round($income * $percent, 8);

                    if ($rebate_amount > 0) {
                        // credit upline flexible_wallet
                        $stmt3 = $conn->prepare("UPDATE users SET flexible_wallet = flexible_wallet + ? WHERE id = ?");
                        $stmt3->execute([$rebate_amount, $uplineId]);

                        // log community income into assets
                        $stmt4 = $conn->prepare("INSERT INTO assets (user_id, type, amount, description, created_at) VALUES (?, 'community_income', ?, ?, NOW())");
                        $stmt4->execute([
                            $uplineId,
                            $rebate_amount,
                            "Community income (Level " . ($levelIndex + 1) . ") from user #$user_id (task $task_id)"
                        ]);
                    }
                }

                // move up
                $current_invite_code = $ref_user['invited_by'];
            }
        }

        // commit
        $conn->commit();

        // success redirect
        $_SESSION['scored_success'] = true;
        header("Location: watch_task.php?task_id=" . urlencode((string)$task_id));
        exit();

    } catch (Exception $e) {
        // rollback on error
        if ($conn->inTransaction()) $conn->rollBack();
        error_log("Error awarding task/rebates: " . $e->getMessage());
        echo "An error occurred while processing the task. Please try again later.";
        exit();
    }
}

// If GET or no action, show page (HTML not repeated here — keep your existing frontend below)
?>

<?php
// watch_task.php
?>
<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Watch Task</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0, maximum-scale=1.0, user-scalable=no, viewport-fit=cover"/>
<script src="https://cdn.tailwindcss.com"></script>
<style>
body {
  background: black;
  color: white;
  font-family: Arial, sans-serif;
  margin: 0;
  overflow-x: hidden;
}

/* Header */
.header {
  background: linear-gradient(180deg, #FFEAB6 0%, #FFFFFF 100%);
  color: #000;
  font-weight: bold;
  text-align: center;
  padding: 12px;
  font-size: 12px;
  position: sticky;
  top: 0;
  z-index: 100;
}
.header .back {
  position: absolute;
  left: 15px;
  font-size: 20px;
  text-decoration: none;
  color: #000;
}

/* Main Video Wrapper */
.main-video-wrapper {
  width: 100%;
  position: relative;
  padding-top: 56%;
  background: #000;
}
.main-video-wrapper iframe,
.main-video-wrapper video {
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  object-fit: cover;
}

/* Thumbnail & Play Button */
.thumb-wrapper {
  position: absolute;
  top: 0;
  left: 0;
  width: 100%;
  height: 100%;
  overflow: hidden;
  cursor: pointer;
  display: flex;
  align-items: center;
  justify-content: center;
}
.thumb-wrapper img {
  width: 100%;
  height: 100%;
  object-fit: cover;
  display: block;
}
.play-btn {
  position: absolute;
  width: 70px;
  height: 70px;
  background: rgba(0,0,0,0.6);
  border-radius: 50%;
  display: flex;
  align-items: center;
  justify-content: center;
}
.play-btn::before {
  content: "";
  display: block;
  margin-left: 5px;
  width: 0;
  height: 0;
  border-left: 25px solid white;
  border-top: 15px solid transparent;
  border-bottom: 15px solid transparent;
}

/* Rating Section */
.rating-comment {
  display: flex;
  flex-direction: column;
  background: #1c1c1c;
  padding: 0.8rem 1rem 1rem;
  border-radius: 0 0 1rem 1rem;
  max-width: 900px;
  margin: 1rem auto;
}
.video-details {
  margin-top: 0.8rem;
  font-size: 0.9rem;
  color: #ccc;
  font-weight: 900; /* Makes the text very bold */
}
.video-details strong {
  font-weight: 900; /* Extra bold */
  background: linear-gradient(180deg, #FFEAB6 0%, #FFFFFF 100%);
  -webkit-background-clip: text;
  -webkit-text-fill-color: transparent;
  display: inline-block;
}
.video-name {
  font-weight: bold;
  font-size: 1rem;
  margin-bottom: 0.4rem;
  color: #fff;
}
.stars {
  display: flex;
  gap: 0.4rem;
  font-size: 1.8rem;
  cursor: pointer;
}
.stars span {
  color: gray;
}
.stars span.selected {
  color: gold;
}
.video-details {
  margin-top: 0.8rem;
  font-size: 0.9rem;
  color: #ccc;
}
.video-details div {
  margin-bottom: 0.3rem;
}

/* Buttons */
.rate-btn {
  margin-top: 10px;
  width: 100%;
  background: #00cc66;
  color: white;
  border: none;
  border-radius: 6px;
  padding: 8px 0;
  cursor: pointer;
  transition: 0.2s;
}
.rate-btn:hover {
  background: #00ee77;
}

/* Toast Notification */
.toast {
  visibility: hidden;
  min-width: 220px;
  background: black;
  color: white;
  text-align: center;
  border-radius: 5px;
  padding: 12px;
  position: fixed;
  top: 50%;
  left: 50%;
  transform: translate(-50%, -50%);
  z-index: 1000;
  font-size: 15px;
}
.toast.show {
  visibility: visible;
  animation: fadein 0.5s, fadeout 0.5s 2s;
}
@keyframes fadein { from { opacity: 0; transform: translate(-50%, -60%); } to { opacity: 1; transform: translate(-50%, -50%); } }
@keyframes fadeout { from { opacity: 1; transform: translate(-50%, -50%); } to { opacity: 0; transform: translate(-50%, -40%); } }

.video-grid {
  display: grid;
  grid-template-columns: repeat(2, 1fr);
  gap: 0.8rem;
  padding: 1rem;
}

.video-item {
  background-color: #1A1A1A;
  border-radius: 0.5rem;
  overflow: hidden;
  cursor: pointer;
  transition: transform 0.2s;
}

.video-item:hover { transform: scale(1.02); }

.video-thumb {
  width: 100%;
  height: 180px;
  object-fit: cover;
}

.video-info {
  padding: 0.6rem;
}

.video-title {
  font-weight: bold;
  font-size: 0.9rem;
  margin-bottom: 0.4rem;
}

.progress-bar {
  background-color: #333;
  border-radius: 0.25rem;
  overflow: hidden;
  height: 6px;
  margin-bottom: 0.3rem;
}

.progress-fill {
  height: 6px;
  width: 80%;
  background: linear-gradient(180deg, #F8D9A0 0%, #FFF4D6 100%);
}

.ticket-label {
  font-size: 0.8rem;
  color: #ccc;
  margin-bottom: 0.5rem;
}

.rating-btn {
  width: 100%;
  background: linear-gradient(180deg, #F8D9A0 0%, #FFF4D6 100%);
  color: #000;
  font-weight: bold;
  text-align: center;
  padding: 0.4rem 0;
  border-radius: 0.25rem;
  font-size: 0.85rem;
  border: none;
  cursor: pointer;
  transition: 0.2s;
}

.rating-btn:hover {
  background: linear-gradient(180deg, #FFEAB6 0%, #FFFFFF 100%);
}
</style>
</head>
<body>

<!-- Header -->
<div class="header">
  <a href="task_vip?vip=<?= $vip_id ?>" class="back">&#8592;</a>
  Watch & Earn
</div>

<!-- Main Video -->
<div class="main-video-wrapper" id="mainVideoWrapper">
<?php
$url = $video['video_url'] ?? '';
$title = htmlspecialchars($video['title'] ?? '', ENT_QUOTES);
$type = htmlspecialchars($video['type'] ?? 'Video', ENT_QUOTES);
$desc = htmlspecialchars($video['description'] ?? '', ENT_QUOTES);

if (!empty($video['thumbnail'])) $main_thumb = $video['thumbnail'];
elseif (strpos($url,'youtube')!==false || strpos($url,'youtu.be')!==false){
  preg_match('/(?:v=|\/)([0-9A-Za-z_-]{11})/',$url,$matches);
  $yt_id = $matches[1] ?? '';
  $main_thumb = "https://img.youtube.com/vi/$yt_id/hqdefault.jpg";
} elseif (strpos($url,'streamable.com')!==false){
  $id = basename($url);
  $main_thumb = "https://cdn-cf-east.streamable.com/image/$id.jpg";
} else $main_thumb='https://img.icons8.com/ios/100/ffffff/video.png';

echo "<div class='thumb-wrapper' id='thumbWrapper'>";
echo "<img id='videoThumb' src='$main_thumb' alt='$title'>";
echo "<div class='play-btn'></div>";
echo "</div>";

if(strpos($url,'youtube')!==false){
  echo "<iframe id='mainVideo' src='https://www.youtube.com/embed/$yt_id?enablejsapi=1&autoplay=1&mute=0' allow='autoplay' allowfullscreen style='display:none;'></iframe>";
} elseif(strpos($url,'streamable.com')!==false){
  $id = basename($url);
  echo "<iframe id='mainVideo' src='https://streamable.com/e/$id?autoplay=1&muted=0' allow='autoplay' allowfullscreen style='display:none;'></iframe>";
} else{
  echo "<video id='mainVideo' src='".htmlspecialchars($url,ENT_QUOTES)."' controls style='display:none;'></video>";
}
?>
</div>

<!-- Rating Section -->
<div class="rating-comment" id="ratingSection">
  <div class="video-name"><?= $title ?></div>
  <div class="stars" id="starRating">
    <span data-value="1">★</span>
    <span data-value="2">★</span>
    <span data-value="3">★</span>
    <span data-value="4">★</span>
    <span data-value="5">★</span>
  </div>



<style>
.video-details div {
  margin-bottom: 0.4rem;
}

.video-details strong {
  /* Keep labels as-is */
}

.video-details span {
  color: #ffffff;       /* Shining white for the write-up values */
  font-weight: bold;    /* Make the text bold */
}
</style>

<div class="video-details">
  <div><strong>Type:</strong> <span><?= $type ?></span></div>
  <div><strong>Title:</strong> <span><?= $title ?></span></div>
  <div><strong>Description:</strong> <span><?= $desc ?></span></div>
</div>
</div>


<form method="post" id="scoreForm" style="display:none;">
  <input type="hidden" name="task_id" value="<?= (int)($task_id ?? 0) ?>">
  <input type="hidden" name="vip_id" value="<?= htmlspecialchars($vip_id ?? '', ENT_QUOTES) ?>">
  <input type="hidden" name="rating" id="ratingInput">
  <button type="submit" class="rate-btn">Score Now</button>
</form>

<div id="toast" class="toast"></div>






<!-- Related Recommendations Heading -->
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css" integrity="sha512-papJk0J8+HgV8p6x8+8T9IQn6pM6x4sK9J+VYgWz1K3k+5ZhZg+zSh84OfbN6lIqWbH7T3xB7t+7xM/lF8UJ0g==" crossorigin="anonymous" referrerpolicy="no-referrer" />

<?php if(!empty($all_videos)): ?>
  <div class="recommendations-header" style="display:flex; align-items:center; gap:0.2rem; padding:0.5rem 1rem; font-size:1.05rem; font-weight:bold; color:#fff;">
    <i class="fa-solid fa-star"></i>
    <span>Related Recommendations</span>
  </div>

  <div class="video-grid-wrapper">
    <div class="video-grid">
      <?php foreach($all_videos as $v):
        if((int)$v['id']===(int)($task_id??0)) continue;
        $v_url = $v['video_url'];
        $v_title = htmlspecialchars($v['title']);
        $tickets = rand(630000, 660000);

        if(!empty($v['thumbnail'])) $thumb=$v['thumbnail'];
        elseif(strpos($v_url,'streamable.com')!==false) $thumb="https://cdn-cf-east.streamable.com/image/".basename($v_url).".jpg";
        elseif(strpos($v_url,'youtube.com')!==false || strpos($v_url,'youtu.be')!==false){
          preg_match('/(?:v=|\/)([0-9A-Za-z_-]{11})/',$v_url,$matches);
          $yt_id = $matches[1] ?? '';
          $thumb = "https://img.youtube.com/vi/$yt_id/hqdefault.jpg";
        } else $thumb='https://img.icons8.com/ios/100/ffffff/video.png';
      ?>
        <div class="video-item" onclick="window.location.href='watch_task.php?task_id=<?= (int)$v['id'] ?>'">
          <img src="<?= $thumb ?>" alt="<?= $v_title ?>" class="video-thumb">
          <div class="video-info">
            <div class="video-title"><?= $v_title ?></div>
            <div class="progress-bar">
              <div class="progress-fill" style="width: <?= rand(60,100) ?>%; background: linear-gradient(180deg, #F8D9A0 0%, #FFF4D6 100%);"></div>
            </div>
            <div class="ticket-label"><?= $tickets ?> Ticket</div>
            <button class="rating-btn" style="background: linear-gradient(180deg, #F8D9A0 0%, #FFF4D6 100%);">Rating immediately</button>
          </div>
        </div>
      <?php endforeach; ?>
    </div>
  </div>
<?php endif; ?>




<script src="https://www.youtube.com/iframe_api"></script>
<script>
const stars = document.querySelectorAll('#starRating span');
const ratingInput = document.getElementById('ratingInput');
const scoreForm = document.getElementById('scoreForm');
const thumbWrapper = document.getElementById('thumbWrapper');
const video = document.getElementById('mainVideo');
const toastEl = document.getElementById('toast');

const alreadyScored = <?= !empty($already_scored)?'true':'false' ?>;
const limitReached = <?= ($scored_today >= (int)($vip['daily_earning_limit']??0))?'true':'false' ?>;

function showToast(msg){
  toastEl.textContent = msg;
  toastEl.classList.add('show');
  setTimeout(()=>toastEl.classList.remove('show'),2500);
}

// Star rating
stars.forEach(star=>{
  star.addEventListener('click',()=>{
    if(alreadyScored || limitReached){
      showToast(alreadyScored?'Already rated this movie':'Earning Limit Reached');
      return;
    }
    const val = parseInt(star.dataset.value,10);
    ratingInput.value = val;
    stars.forEach(s=>s.classList.toggle('selected',parseInt(s.dataset.value)<=val));
    showToast('Success');
    setTimeout(()=>scoreForm.submit(),1200);
  });
});

// Video click to play
thumbWrapper.addEventListener('click',()=>{
  thumbWrapper.style.display='none';
  video.style.display='block';
  if(video.tagName.toLowerCase()==='video'){
    video.play().catch(()=>console.log('HTML5 autoplay blocked'));
  } else {
    video.contentWindow.postMessage('{"event":"command","func":"playVideo","args":""}','*');
    video.contentWindow.postMessage('{"event":"command","func":"unMute","args":""}','*');
  }
});
</script>
</body>
</html>