<?php
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);
error_reporting(E_ALL);
session_start();
include 'db.php';

if (!isset($_SESSION['phone'])) {
    header("Location: login.php");
    exit();
}

$phone = $_SESSION['phone'];

// 🔹 Fetch user data
$stmt = $conn->prepare("SELECT id, flexible_wallet, electronic_wallet, nlc_points, transaction_password 
                        FROM users WHERE phone = ?");
$stmt->execute([$phone]);
$user = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$user) {
    die("User not found.");
}

$user_id = $user['id'];
$flexible_wallet = $user['flexible_wallet'];
$electronic_wallet = $user['electronic_wallet'];
$transaction_password = $user['transaction_password'];

// ✅ Handle AJAX (fund transfer)
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['ajax'])) {
    $amount = floatval($_POST['amount']);
    $tx_pass = $_POST['tx_password'] ?? '';

    // 🔒 Handle missing or invalid password hash
    if (empty($transaction_password) || !password_verify($tx_pass, $transaction_password)) {
        echo json_encode([
            "status" => "error",
            "msg" => "Transaction password not set or incorrect"
        ]);
        exit();
    }

    // 💰 Validate transfer amount
    if ($amount <= 0 || $amount > $flexible_wallet) {
        echo json_encode([
            "status" => "error",
            "msg" => "Invalid amount"
        ]);
        exit();
    }

    try {
        $conn->beginTransaction();

        // 🔁 Update wallets
        $stmt1 = $conn->prepare("
            UPDATE users 
            SET flexible_wallet = flexible_wallet - ?, 
                electronic_wallet = electronic_wallet + ? 
            WHERE id = ?
        ");
        $stmt1->execute([$amount, $amount, $user_id]);

        // 🧾 Record exchange if table exists
        $tableCheck = $conn->query("SHOW TABLES LIKE 'exchange_records'");
        if ($tableCheck && $tableCheck->rowCount() > 0) {
            $stmt2 = $conn->prepare("
                INSERT INTO exchange_records (user_id, amount, created_at) 
                VALUES (?, ?, NOW())
            ");
            $stmt2->execute([$user_id, $amount]);
        }

        $conn->commit();

        // 🔄 Fetch updated balances
        $stmt = $conn->prepare("
            SELECT flexible_wallet, electronic_wallet 
            FROM users WHERE id = ?
        ");
        $stmt->execute([$user_id]);
        $updated = $stmt->fetch(PDO::FETCH_ASSOC);

        echo json_encode([
            "status" => "success",
            "msg" => "Transfer Successful!",
            "flexible_wallet" => number_format($updated['flexible_wallet'], 4),
            "electronic_wallet" => number_format($updated['electronic_wallet'], 4)
        ]);
        exit();

    } catch (Exception $e) {
        $conn->rollBack();
        echo json_encode([
            "status" => "error",
            "msg" => "Transfer failed: " . $e->getMessage()
        ]);
        exit();
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8">
<title>Fund Transfer</title>
<meta name="viewport" content="width=device-width, initial-scale=1.0">
<style>
  body {
    font-family: Arial, sans-serif;
    margin: 0;
    background: #111;
    color: #fff;
  }

  .header {
    background: linear-gradient(to bottom, #fff8e3, #ffd78a);  
    text-align: center;
    padding: 14px 0;
    font-weight: bold;
    font-size: 16px;
    position: relative;
  }

  .back-btn {
    position: absolute;
    left: 10px;
    top: 50%;
    transform: translateY(-50%);
    color: #fff;
    text-decoration: none;
    font-size: 20px;
  }

  .container { padding: 20px; }

  .card {
    background: #1c1c1c;
    border-radius: 10px;
    padding: 15px;
    margin-bottom: 20px;
  }

  .card label {
    font-size: 14px;
    color: #ccc;
    display: block;
    margin-bottom: 8px;
  }

  .value-box {
    background: #2a2a2a;
    padding: 12px;
    border-radius: 8px;
    font-size: 15px;
    font-weight: bold;
  }

  input[type="number"] {
    width: 100%;
    padding: 12px;
    border: none;
    border-radius: 8px;
    background: #2a2a2a;
    color: #fff;
    font-size: 15px;
  }

  input::placeholder { color: #777; }

  .btn {
    width: 100%;
    padding: 14px;
    background: linear-gradient(to bottom, #fff8e3, #ffd78a);  
    border: none;
    border-radius: 30px;
    font-size: 16px;
    font-weight: bold;
    color: #111;
    cursor: pointer;
  }

  .btn:hover { opacity: 0.9; }

  .popup {
    position: fixed;
    left: 0; right: 0; bottom: -100%;
    background: #1c1c1c;
    border-radius: 20px 20px 0 0;
    transition: bottom 0.3s ease;
    padding: 20px;
    z-index: 1000;
  }

  .popup.active { bottom: 0; }

  .pass-inputs {
    display: flex;
    justify-content: space-between;
    margin: 20px 0;
  }

  .pass-inputs input {
    width: 40px;
    height: 50px;
    text-align: center;
    font-size: 20px;
    border: none;
    border-radius: 8px;
    background: #2a2a2a;
    color: #fff;
  }

  .toast {
    visibility: hidden;
    background: #000;
    color: #fff;
    text-align: center;
    border-radius: 8px;
    padding: 8px 12px;
    position: fixed;
    z-index: 2000;
    left: 50%;
    bottom: 60px;
    font-size: 12px;
    transform: translateX(-50%);
    opacity: 0;
    transition: opacity 0.3s ease, bottom 0.3s ease;
  }

  .toast.show {
    visibility: visible;
    opacity: 1;
    bottom: 80px;
  }
</style>
</head>
<body>

<div class="header">
  <a href="#" class="back-btn">←</a>
  Fund Transfer
</div>

<div class="container">
  <div class="card">
    <label>Income Account</label>
    <div class="value-box" id="income-balance">
      <?php echo number_format($flexible_wallet, 4); ?> USD
    </div>
  </div>

  <div class="card">
    <label>Amount</label>
    <input type="number" id="amount" placeholder="Enter amount">
  </div>

  <div class="card">
    <label>Transferable Funds</label>
    <div class="value-box" id="transferable-balance">
      <?php echo number_format($electronic_wallet, 4); ?> USD
    </div>
  </div>

  <div class="tip">
    Transfer income account funds to the capital account, usable for package purchases only.
  </div>

  <button class="btn" onclick="openPopup()">Transfer</button>
</div>

<!-- Password Popup -->
<div class="popup" id="popup">
  <h3>Please enter the transaction password</h3>
  <div class="pass-inputs">
    <input maxlength="1" oninput="moveNext(this,0)">
    <input maxlength="1" oninput="moveNext(this,1)">
    <input maxlength="1" oninput="moveNext(this,2)">
    <input maxlength="1" oninput="moveNext(this,3)">
    <input maxlength="1" oninput="moveNext(this,4)">
    <input maxlength="1" oninput="moveNext(this,5)">
  </div>
  <button class="btn" onclick="submitTransfer()">Confirm</button>
</div>

<!-- Toast -->
<div id="toast" class="toast"></div>

<script>
function openPopup() {
  document.getElementById('popup').classList.add('active');
}

function moveNext(input, index) {
  if (input.value.length === 1) {
    let next = input.parentElement.children[index + 1];
    if (next) next.focus();
  }
}

function showToast(msg) {
  const toast = document.getElementById("toast");
  toast.innerHTML = msg;
  toast.className = "toast show";
  setTimeout(() => toast.className = toast.className.replace("show", ""), 2000);
}

async function submitTransfer() {
  const digits = document.querySelectorAll('.pass-inputs input');
  let pass = '';
  digits.forEach(d => pass += d.value);
  const amount = document.getElementById('amount').value;

  if (pass.length !== 6) {
    showToast("Enter 6-digit password");
    return;
  }

  if (!amount || amount <= 0) {
    showToast("Enter a valid amount");
    return;
  }

  // 🔥 Send AJAX to backend
  const formData = new FormData();
  formData.append('ajax', '1');
  formData.append('amount', amount);
  formData.append('tx_password', pass);

  const res = await fetch('', { method: 'POST', body: formData });
  const data = await res.json();

  if (data.status === 'success') {
    document.getElementById('income-balance').innerText = data.flexible_wallet + ' USD';
    document.getElementById('transferable-balance').innerText = data.electronic_wallet + ' USD';
    showToast(data.msg);
  } else {
    showToast(data.msg);
  }

  document.getElementById('popup').classList.remove('active');
  document.querySelectorAll('.pass-inputs input').forEach(i => i.value = "");
}
</script>

</body>
</html>